/* THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR REPRESENTATIONS,
 * EITHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING ANY IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY
 * OR COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE. TI DISCLAIMS
 * ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET POSSESSION, AND NON-INFRINGEMENT
 * OF ANY THIRD PARTY INTELLECTUAL PROPERTY RIGHTS WITH REGARD TO THE PROGRAM OR
 * YOUR USE OF THE PROGRAM.
 * IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL, CONSEQUENTIAL OR
 * INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY THEORY OF LIABILITY AND WHETHER OR
 * NOT TI HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY
 * OUT OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM.  EXCLUDED
 * DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF REMOVAL OR REINSTALLATION,
 * COMPUTER TIME, LABOR COSTS, LOSS OF GOODWILL, LOSS OF PROFITS, LOSS OF
 * SAVINGS, OR LOSS OF USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S
 * AGGREGATE LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF YOUR USE OF THE
 * PROGRAM EXCEED FIVE HUNDRED DOLLARS (U.S.$500).
 * Unless otherwise stated, the Program written and copyrighted by Texas
 * Instruments is distributed as "freeware".  You may, only under TI's copyright
 * in the Program, use and modify the Program without any charge or restriction.
 * You may distribute to third parties, provided that you transfer a copy of this
 * license to the third party and the third party agrees to these terms by its
 * first use of the Program. You must reproduce the copyright notice and any
 * other legend of ownership on each copy or partial copy, of the Program.
 * You acknowledge and agree that the Program contains copyrighted material,
 * trade secrets and other TI proprietary information and is protected by
 * copyright laws, international copyright treaties, and trade secret laws, as
 * well as other intellectual property laws.  To protect TI's rights in the
 * Program, you agree not to decompile, reverse engineer, disassemble or
 * otherwise translate any object code versions of the Program to a
 * human-readable form.  You agree that in no event will you alter, remove or
 * destroy any copyright notice included in the Program.  TI reserves all rights
 * not specifically granted under this license. Except as specifically provided
 * herein, nothing in this agreement shall be construed as conferring by
 * implication, estoppel, or otherwise, upon you, any license or other right
 * under any TI patents, copyrights or trade secrets.
 * You may not use the Program in non-TI devices. */


//******************************************************************************
//
//  lcd.c
//  Control of a HD47780 compatible LCD
//
//  Matthias Ulmann, Design Services - EMEA Power Management
//  Texas Instruments, Inc.
//
//******************************************************************************


#include "main.h"



#define lcd_en_delay()			delay_us(2);		// min. 450ns needed
#define lcd_en_high()			set_bit(LCD_EN_POUT, LCD_EN_BIT)
#define lcd_en_low()			clear_bit(LCD_EN_POUT, LCD_EN_BIT)
#define lcd_en_toggle()			toggle_en()
#define lcd_rw_high()			set_bit(LCD_RW_POUT, LCD_RW_BIT)
#define lcd_rw_low()			clear_bit(LCD_RW_POUT, LCD_RW_BIT)
#define lcd_rs_high()			set_bit(LCD_RS_POUT, LCD_RS_BIT)
#define lcd_rs_low()			clear_bit(LCD_RS_POUT, LCD_RS_BIT)


#if LCD_LINES==1
#define LCD_FUNCTION_DEFAULT    LCD_FUNCTION_4BIT_1LINE 
#else
#define LCD_FUNCTION_DEFAULT    LCD_FUNCTION_4BIT_2LINES 
#endif



// Reverse string s in place
 void reverse(char s[])
 {
 	int i, j;
 	char c;
 	
 	for (i = 0, j = strlen(s)-1; i<j; i++, j--)
 	{
 		c = s[i];
 		s[i] = s[j];
 		s[j] = c;
 	}
 }



// Convert n to characters in s
 void itoa(int n, char s[])
 {
 	int i, sign;
 	
 	if ((sign = n) < 0)		// record sign
 		n = -n;				// make n positive
 	i = 0;
 	do {	// generate digits in reverse order
 		s[i++] = n % 10 + '0';	// get next digit
 		}
 	 while ((n /= 10) > 0);		// delete it
 	 
 	 if (sign < 0) s[i++] = '-';
 	 s[i] = '\0';
 	 reverse(s);
 }



// toggle Enable Pin to initiate write
static void toggle_en(void)
{
    lcd_en_high();
    lcd_en_delay();
    lcd_en_low();
}



/*************************************************************************
Low-level function to write byte to LCD controller
Input:    data   byte to write to LCD
          rs     1: write data    
                 0: write instruction
Returns:  none
*************************************************************************/
static void lcd_write(unsigned int data, unsigned int rs) 
{
	if (rs)
	{
		lcd_rs_high();		// write data (RS=1, RW=0)
	}
	else
	{
		lcd_rs_low();		// write instruction (RS=0, RW=0)
	}
    
    lcd_rw_low();
    
    // set D0..D3 as output
    set_bit(LCD_DATA0_DDR, LCD_DATA0_BIT);
	set_bit(LCD_DATA1_DDR, LCD_DATA1_BIT);
	set_bit(LCD_DATA2_DDR, LCD_DATA2_BIT);
	set_bit(LCD_DATA3_DDR, LCD_DATA3_BIT);
    
    // set I/O function
	clear_bit(LCD_DATA0_PSEL, LCD_DATA0_BIT);
	clear_bit(LCD_DATA1_PSEL, LCD_DATA1_BIT);
	clear_bit(LCD_DATA2_PSEL, LCD_DATA2_BIT);
	clear_bit(LCD_DATA3_PSEL, LCD_DATA3_BIT);
       
    // output high nibble first
    clear_bit(LCD_DATA3_POUT, LCD_DATA3_BIT);
    clear_bit(LCD_DATA2_POUT, LCD_DATA2_BIT);
    clear_bit(LCD_DATA1_POUT, LCD_DATA1_BIT);
    clear_bit(LCD_DATA0_POUT, LCD_DATA0_BIT);
    if(data & 0x80) set_bit(LCD_DATA3_POUT, LCD_DATA3_BIT);
    if(data & 0x40) set_bit(LCD_DATA2_POUT, LCD_DATA2_BIT);
    if(data & 0x20) set_bit(LCD_DATA1_POUT, LCD_DATA1_BIT);
    if(data & 0x10) set_bit(LCD_DATA0_POUT, LCD_DATA0_BIT);   
    
    lcd_en_toggle();
    
    // output low nibble
    clear_bit(LCD_DATA3_POUT, LCD_DATA3_BIT);
    clear_bit(LCD_DATA2_POUT, LCD_DATA2_BIT);
    clear_bit(LCD_DATA1_POUT, LCD_DATA1_BIT);
    clear_bit(LCD_DATA0_POUT, LCD_DATA0_BIT);
    if(data & 0x08) set_bit(LCD_DATA3_POUT, LCD_DATA3_BIT);
    if(data & 0x04) set_bit(LCD_DATA2_POUT, LCD_DATA2_BIT);
    if(data & 0x02) set_bit(LCD_DATA1_POUT, LCD_DATA1_BIT);
    if(data & 0x01) set_bit(LCD_DATA0_POUT, LCD_DATA0_BIT);
    
    lcd_en_toggle();
    
    // all data pins high (inactive)
    set_bit(LCD_DATA0_POUT, LCD_DATA0_BIT);
    set_bit(LCD_DATA1_POUT, LCD_DATA1_BIT);
    set_bit(LCD_DATA2_POUT, LCD_DATA2_BIT);
    set_bit(LCD_DATA3_POUT, LCD_DATA3_BIT);
}



/*************************************************************************
Low-level function to read byte from LCD controller
Input:    rs     1: read data    
                 0: read busy flag / address counter
Returns:  byte read from LCD controller
*************************************************************************/
static unsigned int lcd_read(unsigned int rs) 
{
	unsigned int data;
	
	
	if(rs) lcd_rs_high();		// RS=1: read data
	else lcd_rs_low();			// RS=0: read busy flag
	
	lcd_rw_high();			// RW=1  read mode
    
    
    // set D0..D3 as input
    clear_bit(LCD_DATA0_DDR, LCD_DATA0_BIT);
    clear_bit(LCD_DATA1_DDR, LCD_DATA1_BIT);
    clear_bit(LCD_DATA2_DDR, LCD_DATA2_BIT);
    clear_bit(LCD_DATA3_DDR, LCD_DATA3_BIT);
    
    // disable pullups for D0..D3
    clear_bit(LCD_DATA0_REN, LCD_DATA0_BIT);
    clear_bit(LCD_DATA1_REN, LCD_DATA1_BIT);
    clear_bit(LCD_DATA2_REN, LCD_DATA2_BIT);
    clear_bit(LCD_DATA3_REN, LCD_DATA3_BIT);
    
    // set I/O function
	clear_bit(LCD_DATA0_PSEL, LCD_DATA0_BIT);
	clear_bit(LCD_DATA1_PSEL, LCD_DATA1_BIT);
	clear_bit(LCD_DATA2_PSEL, LCD_DATA2_BIT);
	clear_bit(LCD_DATA3_PSEL, LCD_DATA3_BIT);
	
	
	// read high nibble first
	lcd_en_high();
    lcd_en_delay();        
    data = 0;
    
    if ( LCD_DATA0_PIN & LCD_DATA0_BIT )	data |= 0x10;
    if ( LCD_DATA1_PIN & LCD_DATA1_BIT )	data |= 0x20;
    if ( LCD_DATA2_PIN & LCD_DATA2_BIT )	data |= 0x40;
    if ( LCD_DATA3_PIN & LCD_DATA3_BIT )	data |= 0x80;
    lcd_en_low();
    
    lcd_en_delay();
    
    // read low nibble
    lcd_en_high();
    lcd_en_delay();
    if ( LCD_DATA0_PIN & LCD_DATA0_BIT )	data |= 0x01;
    if ( LCD_DATA1_PIN & LCD_DATA1_BIT )	data |= 0x02;
    if ( LCD_DATA2_PIN & LCD_DATA2_BIT )	data |= 0x04;
    if ( LCD_DATA3_PIN & LCD_DATA3_BIT )	data |= 0x08;        
    lcd_en_low();
    
    return data;
}



/*************************************************************************
loops while lcd is busy, returns address counter
*************************************************************************/
static unsigned int lcd_waitbusy(void)
{
	// wait until busy flag is cleared
	while ( (lcd_read(0)) & (1<<LCD_BUSY)) {}
    
    // the address counter is updated 4us after the busy flag is cleared
    delay_us(5);
    
    // now read and return the address counter
    return (lcd_read(0));
}



/*************************************************************************
Move cursor to the start of next line or to the first line if the cursor 
is already on the last line.
*************************************************************************/
static inline void lcd_newline(unsigned int pos)
{
	register unsigned int addressCounter;
	
	#if LCD_LINES==1
		addressCounter = 0;
	#endif
	
	#if LCD_LINES==2
		if ( pos < (LCD_START_LINE2) )
			addressCounter = LCD_START_LINE2;
		else
			addressCounter = LCD_START_LINE1;
	#endif
	
	#if LCD_LINES==4
	#if KS0073_4LINES_MODE
		if ( pos < LCD_START_LINE2 )
        	addressCounter = LCD_START_LINE2;
    	else if ( (pos >= LCD_START_LINE2) && (pos < LCD_START_LINE3) )
        	addressCounter = LCD_START_LINE3;
    	else if ( (pos >= LCD_START_LINE3) && (pos < LCD_START_LINE4) )
        	addressCounter = LCD_START_LINE4;
    	else 
        	addressCounter = LCD_START_LINE1;
	#else
    	if ( pos < LCD_START_LINE3 )
        	addressCounter = LCD_START_LINE2;
    	else if ( (pos >= LCD_START_LINE2) && (pos < LCD_START_LINE4) )
        	addressCounter = LCD_START_LINE3;
    	else if ( (pos >= LCD_START_LINE3) && (pos < LCD_START_LINE2) )
        	addressCounter = LCD_START_LINE4;
    	else 
        	addressCounter = LCD_START_LINE1;
	#endif
	#endif
	
    lcd_command((1<<LCD_DDRAM)+addressCounter);
}



/*************************************************************************
Send LCD controller instruction command
Input:   instruction to send to LCD controller, see HD44780 data sheet
*************************************************************************/
void lcd_command(unsigned int cmd)
{
	lcd_waitbusy();
    lcd_write(cmd,0);
}



/*************************************************************************
Send data byte to LCD controller 
Input:   data to send to LCD controller, see HD44780 data sheet
*************************************************************************/
void lcd_data(unsigned int data)
{
    lcd_waitbusy();
    lcd_write(data,1);
}



/*************************************************************************
Set cursor to specified position
Input:    x  horizontal position  (0: left most position)
          y  vertical position    (0: first line)
*************************************************************************/
void lcd_gotoxy(unsigned int x, unsigned int y)
{
	#if LCD_LINES==1
    	lcd_command((1<<LCD_DDRAM)+LCD_START_LINE1+x);
	#endif
	
	#if LCD_LINES==2
    	if ( y==0 ) 
        	lcd_command((1<<LCD_DDRAM)+LCD_START_LINE1+x);
    	else
        	lcd_command((1<<LCD_DDRAM)+LCD_START_LINE2+x);
	#endif
	
	#if LCD_LINES==4
    	if ( y==0 )
        	lcd_command((1<<LCD_DDRAM)+LCD_START_LINE1+x);
    	else if ( y==1)
        	lcd_command((1<<LCD_DDRAM)+LCD_START_LINE2+x);
    	else if ( y==2)
        	lcd_command((1<<LCD_DDRAM)+LCD_START_LINE3+x);
    	else /* y==3 */
        	lcd_command((1<<LCD_DDRAM)+LCD_START_LINE4+x);
	#endif
}



/*************************************************************************
Clear display and set cursor to home position
*************************************************************************/
void lcd_clrscr(void)
{
	lcd_command(1<<LCD_CLR);
}



/*************************************************************************
Set cursor to home position
*************************************************************************/
void lcd_home(void)
{
    lcd_command(1<<LCD_HOME);
}



/*************************************************************************
Display character at current cursor position 
Input:    character to be displayed                                       
*************************************************************************/
void lcd_putc(char c)
{
    unsigned int pos;

    pos = lcd_waitbusy();	// read busy-flag and address counter
    if (c=='\n')
    {
    	lcd_newline(pos);
    }
    else
    {
    #if LCD_WRAP_LINES==1
		#if LCD_LINES==1
        	if ( pos == LCD_START_LINE1+LCD_DISP_LENGTH )
        	{
        		lcd_write((1<<LCD_DDRAM)+LCD_START_LINE1,0);
        	}
        
        #elif LCD_LINES==2
        	if ( pos == LCD_START_LINE1+LCD_DISP_LENGTH )
        	{
        		lcd_write((1<<LCD_DDRAM)+LCD_START_LINE2,0);
        	}
        	else if ( pos == LCD_START_LINE2+LCD_DISP_LENGTH 
        	{
        		lcd_write((1<<LCD_DDRAM)+LCD_START_LINE1,0);
        	}
	}
	
	#elif LCD_LINES==4
        if ( pos == LCD_START_LINE1+LCD_DISP_LENGTH )
        {
        	lcd_write((1<<LCD_DDRAM)+LCD_START_LINE2,0);
        }    
        else if ( pos == LCD_START_LINE2+LCD_DISP_LENGTH )
        {
            lcd_write((1<<LCD_DDRAM)+LCD_START_LINE3,0);
        }
        else if ( pos == LCD_START_LINE3+LCD_DISP_LENGTH )
        {
            lcd_write((1<<LCD_DDRAM)+LCD_START_LINE4,0);
        }
        else if ( pos == LCD_START_LINE4+LCD_DISP_LENGTH )
        {
            lcd_write((1<<LCD_DDRAM)+LCD_START_LINE1,0);
        }
	#endif
    
    lcd_waitbusy();
	#endif
	lcd_write(c, 1);
    }
}



/*************************************************************************
Display string without auto linefeed 
Input:    string to be displayed
*************************************************************************/
void lcd_puts(const char *s)
{
	register char c;
	
	while ( (c = *s++) )
	{
        lcd_putc(c);
    }
}



// Display an integer
// i: integer to be displayed
void lcd_puti(unsigned int var_integer)
{
	char var_string[3];
	
	// convert integer to string
	itoa(var_integer, var_string);
	
	// print string
	lcd_puts(var_string);
}



/*************************************************************************
Initialize display and select type of cursor 
Input:    dispAttr LCD_DISP_OFF            display off
                   LCD_DISP_ON             display on, cursor off
                   LCD_DISP_ON_CURSOR      display on, cursor on
                   LCD_DISP_CURSOR_BLINK   display on, cursor on flashing
*************************************************************************/
void lcd_init(unsigned int dispAttr)
{
	// set D0..D3 as output
    set_bit(LCD_DATA0_DDR, LCD_DATA0_BIT);
	set_bit(LCD_DATA1_DDR, LCD_DATA1_BIT);
	set_bit(LCD_DATA2_DDR, LCD_DATA2_BIT);
	set_bit(LCD_DATA3_DDR, LCD_DATA3_BIT);
    
    // set I/O function for D0..D3
	clear_bit(LCD_DATA0_PSEL, LCD_DATA0_BIT);
	clear_bit(LCD_DATA1_PSEL, LCD_DATA1_BIT);
	clear_bit(LCD_DATA2_PSEL, LCD_DATA2_BIT);
	clear_bit(LCD_DATA3_PSEL, LCD_DATA3_BIT);
	
	// set RS/RW/E as output
	set_bit(LCD_RS_DDR, LCD_RS_BIT);
	set_bit(LCD_RW_DDR, LCD_RW_BIT);
	set_bit(LCD_EN_DDR, LCD_EN_BIT);
	
	// set I/O function for RS/RW/E
	clear_bit(LCD_RS_PSEL, LCD_RS_BIT);
	clear_bit(LCD_RW_PSEL, LCD_RW_BIT);
	clear_bit(LCD_EN_PSEL, LCD_EN_BIT);
	
	delay_us(16000);		// wait 16ms or more after power-on
	
	// initial write to LCD is 8bit
    set_bit(LCD_DATA1_POUT, LCD_DATA1_BIT);
    set_bit(LCD_DATA0_POUT, LCD_DATA0_PIN);
    lcd_en_toggle();
    delay_us(5000); 		// delay, busy flag can't be checked here
    
    // repeat last command
    lcd_en_toggle();      
    delay_us(120);			// delay, busy flag can't be checked here
    
    // repeat last command a third time
    lcd_en_toggle();      
    delay_us(120);			// delay, busy flag can't be checked here
    
    // now configure for 4bit mode
    clear_bit(LCD_DATA0_POUT, LCD_DATA0_BIT);
    lcd_en_toggle();
    delay_us(120);			// some displays need this additional delay
    
	lcd_command(LCD_FUNCTION_DEFAULT);		// function set: display lines
    lcd_command(LCD_DISP_OFF);				// display off
    lcd_clrscr();							// display clear 
    lcd_command(LCD_MODE_DEFAULT);			// set entry mode
    lcd_command(dispAttr);					// display/cursor control
}
